/* eslint-disable @typescript-eslint/no-non-null-assertion */
/// <reference lib="dom" />
// importing default is needed for ESM compatibility
// default-index is a CJS file, so named exports are not resolved
// the types create a separate resolution issue, so they are still imported by name
import PrismaDefault from "@prisma/client/scripts/default-index.js";
import { compareSemVer } from "./semver.js";
import { getUserAgent } from "./user-agent.js";
const EXTENSION_NAME = "@prisma/extension-accelerate";
export const FETCH_FAILURE_MESSAGE = "Unable to connect to the Accelerate API. This may be due to a network or DNS issue. Please check your connection and the Accelerate connection string. For details, visit https://www.prisma.io/docs/accelerate/troubleshoot.";
function makeWithCacheHeaders(fetcher) {
    const userAgent = getUserAgent();
    let machineHint = undefined;
    return async (params) => {
        const { args } = params;
        const { cacheStrategy, __accelerateInfo = false, ...rest } = args;
        let info = null;
        const { __internalParams, query } = params;
        __internalParams.customDataProxyFetch = () => {
            return async (url, options) => {
                const cacheControl = new Array();
                if (typeof cacheStrategy?.ttl === "number") {
                    cacheControl.push(`max-age=${cacheStrategy.ttl}`);
                }
                if (typeof cacheStrategy?.swr === "number") {
                    cacheControl.push(`stale-while-revalidate=${cacheStrategy.swr}`);
                }
                const cacheTags = cacheStrategy?.tags?.join(",") ?? "";
                options.headers = {
                    ...options.headers,
                    "cache-control": cacheControl.length > 0 ? cacheControl.join(",") : `no-cache`,
                    "user-agent": userAgent,
                    ...(cacheTags.length > 0
                        ? { "accelerate-cache-tags": cacheTags }
                        : {}),
                };
                if (machineHint) {
                    options.headers["accelerate-query-engine-jwt"] = machineHint;
                }
                try {
                    const response = await fetcher(url, options);
                    info = {
                        cacheStatus: response.headers.get("accelerate-cache-status"),
                        lastModified: new Date(response.headers.get("last-modified") ?? ""),
                        region: response.headers.get("cf-ray")?.split("-")[1] ?? "unspecified",
                        requestId: response.headers.get("cf-ray") ?? "unspecified",
                        signature: response.headers.get("accelerate-signature") ?? "unspecified",
                    };
                    machineHint =
                        response.headers.get("accelerate-query-engine-jwt") ?? undefined;
                    return response;
                }
                catch (e) {
                    throw new Error(FETCH_FAILURE_MESSAGE);
                }
            };
        };
        if (__accelerateInfo) {
            const data = await query(rest, __internalParams);
            return { data, info };
        }
        else {
            return query(rest, __internalParams);
        }
    };
}
export function makeAccelerateExtension(fetcher) {
    // ctx.$parent was added in 5.1.0 to support iTx in extensions
    const enableCtxParent = compareSemVer("5.1.0", PrismaDefault.Prisma.prismaVersion.client) >= 0;
    return PrismaDefault.Prisma.defineExtension((client) => {
        const withCacheHeaders = makeWithCacheHeaders(fetcher);
        // api key is extracted from config during engine start
        const apiKeyPromise = client._engine
            .start()
            .then(() => client._engine.apiKey?.());
        const xclient = client.$extends({
            name: EXTENSION_NAME,
            query: {
                $allModels: {
                    // also apply withCacheHeaders to mutations for machine hint benefit
                    $allOperations: withCacheHeaders,
                },
            },
        });
        return xclient.$extends({
            name: EXTENSION_NAME,
            client: {
                $accelerate: {
                    /**
                     * Initiates an invalidation request for the specified cache tag values.
                     *
                     * A tag may only contain alphanumeric characters and underscores.
                     * Each tag may contain a maximum of 64 characters.
                     * A maximum of 5 tags may be invalidated per call.
                     */
                    invalidate: async (input) => {
                        const apiKey = await apiKeyPromise;
                        if (!apiKey) {
                            return { requestId: "unspecified" };
                        }
                        let response;
                        try {
                            response = await fetcher(`https://accelerate.prisma-data.net/invalidate`, {
                                method: "POST",
                                headers: {
                                    authorization: `Bearer ${apiKey}`,
                                    "content-type": "application/json",
                                },
                                body: JSON.stringify(input),
                            });
                        }
                        catch (e) {
                            throw new Error(FETCH_FAILURE_MESSAGE);
                        }
                        if (!response?.ok) {
                            const body = await response.text();
                            throw new Error(`Failed to invalidate Accelerate cache. Response was ${response.status} ${response.statusText}. ${body}`);
                        }
                        // discard the response body to avoid memory leaks on some runtimes
                        void response.body?.cancel();
                        return {
                            requestId: response.headers.get("cf-ray") ?? "unspecified",
                        };
                    },
                },
            },
            model: {
                $allModels: {
                    // TODO: these functions are repetitive. Is there a type we can use to generic this?
                    // TODO: can we define these in a map that ensures query and model overrides stay in sync/
                    aggregate(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.aggregate(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.aggregate({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    count(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.count(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.count({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    findFirst(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.findFirst(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.findFirst({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    findFirstOrThrow(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.findFirstOrThrow(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.findFirstOrThrow({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    findMany(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.findMany(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.findMany({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    findUnique(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.findUnique(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.findUnique({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    findUniqueOrThrow(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.findUniqueOrThrow(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.findUniqueOrThrow({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                    groupBy(args) {
                        const ctx = PrismaDefault.Prisma.getExtensionContext(this);
                        const model = enableCtxParent
                            ? ctx.$parent[ctx.$name]
                            : xclient[ctx.name];
                        const prismaPromise = model.groupBy(args);
                        return Object.assign(prismaPromise, {
                            withAccelerateInfo() {
                                return model.groupBy({
                                    ...args,
                                    __accelerateInfo: true,
                                });
                            },
                        });
                    },
                },
            },
        });
    });
}
