/**
 * @license Copyright (c) 2003-2025, CKSource Holding sp. z o.o. All rights reserved.
 * For licensing, see LICENSE.md or https://ckeditor.com/legal/ckeditor-licensing-options
 */
import React, { type PropsWithChildren, type ReactElement } from 'react';
import { type InitializedContextEditorsConfig } from './useInitializedCKEditorsMap.js';
import type { ContextWatchdog, WatchdogConfig, Context, ContextConfig } from 'ckeditor5';
export declare const ContextWatchdogContext: React.Context<ContextWatchdogValue<Context> | null>;
/**
 * Custom hook that returns the CKEditor Watchdog context value.
 */
export declare const useCKEditorWatchdogContext: () => ContextWatchdogValue | null;
/**
 * A React component that provides a context for CKEditor.
 */
declare const CKEditorContext: <TContext extends Context = Context>(props: Props<TContext>) => ReactElement | null;
/**
 * Checks if the given object is of type ContextWatchdogValue.
 *
 * @param obj The object to be checked.
 * @returns True if the object is of type ContextWatchdogValue, false otherwise.
 */
export declare const isContextWatchdogValue: (obj: any) => obj is ContextWatchdogValue;
/**
 * Checks if the provided object is a context watchdog value with the specified status.
 */
export declare const isContextWatchdogValueWithStatus: <S extends ContextWatchdogValueStatus>(status: S) => (obj: any) => obj is ExtractContextWatchdogValueByStatus<S>;
/**
 * Checks if the context watchdog is currently initializing.
 */
export declare const isContextWatchdogInitializing: (obj: any) => obj is ExtractContextWatchdogValueByStatus<"initializing">;
/**
 * Checks if the provided object is a fully initialized context watchdog value. It prevents race conditions between
 * watchdog state that is not fully synchronized with the context state. For example, the watchdog state can be 'destroyed'
 * while the context is still being initialized because context setState is pending.
 */
export declare const isContextWatchdogReadyToUse: (obj: any) => obj is ExtractContextWatchdogValueByStatus<"initialized">;
/**
 * Represents the value of the ContextWatchdog in the CKEditor context.
 */
export type ContextWatchdogValue<TContext extends Context = Context> = {
    status: 'initializing';
} | {
    status: 'initialized';
    watchdog: ContextWatchdog<TContext>;
} | {
    status: 'error';
    error: ErrorDetails;
};
/**
 * Represents the status of the ContextWatchdogValue.
 */
export type ContextWatchdogValueStatus = ContextWatchdogValue['status'];
/**
 * Extracts a specific type of `ContextWatchdogValue` based on its status.
 */
export type ExtractContextWatchdogValueByStatus<S extends ContextWatchdogValueStatus> = Extract<ContextWatchdogValue, {
    status: S;
}>;
/**
 * Props for the CKEditorContext component.
 */
export type Props<TContext extends Context> = PropsWithChildren & Pick<InitializedContextEditorsConfig<TContext>, 'onChangeInitializedEditors'> & {
    id?: string;
    isLayoutReady?: boolean;
    context?: {
        create(...args: any): Promise<TContext>;
    };
    contextWatchdog: typeof ContextWatchdog<TContext>;
    watchdogConfig?: WatchdogConfig;
    config?: ContextConfig;
    onReady?: (context: TContext, watchdog: ContextWatchdog<TContext>) => void;
    onError?: (error: Error, details: ErrorDetails) => void;
};
type ErrorDetails = {
    phase: 'initialization' | 'runtime';
    willContextRestart: boolean;
};
export default CKEditorContext;
