import { Awaitable } from '../../types/Awaitable.js';
import { InjectScriptProps } from '../../utils/injectScript.js';
/**
 * Loads pack of resources (scripts and stylesheets) and returns the exported global variables (if any).
 *
 * @param pack The pack of resources to load.
 * @returns A promise that resolves with the exported global variables.
 * @example
 *
 * ```ts
 * const ckeditor = await loadCKCdnResourcesPack<ClassicEditor>( {
 * 	scripts: [
 * 		'https://cdn.ckeditor.com/ckeditor5/30.0.0/classic/ckeditor.js'
 * 	],
 * 	checkPluginLoaded: () => ( window as any ).ClassicEditor
 * } );
 * ```
 */
export declare function loadCKCdnResourcesPack<P extends CKCdnResourcesPack<any>>(pack: P): Promise<InferCKCdnResourcesPackExportsType<P>>;
/**
 * Normalizes the CKCdnResourcesPack configuration to the advanced format.
 *
 * @param pack The pack of resources to normalize.
 * @returns The normalized pack of resources.
 */
export declare function normalizeCKCdnResourcesPack<R = any>(pack: CKCdnResourcesPack<R>): CKCdnResourcesAdvancedPack<R>;
/**
 * A pack of resources to load (scripts and stylesheets) and the exported global variables.
 */
export type CKCdnResourcesPack<R = any> = CKCdnResourcesAdvancedPack<R> | CKCdnResourcesBasicUrlsPack | CKCdnResourcesLocalPack<R>;
/**
 * A pack of resources to load as an async function that results with UMD module.
 *
 * @example
 * ```ts
 * const pack = async () => import( './your-module' );
 * ```
 */
type CKCdnResourcesLocalPack<R> = () => Awaitable<R>;
/**
 * A pack of resources to load (scripts and stylesheets). In such configuration, the exported global variable
 * might be available but it's not guaranteed.
 *
 * @example
 * ```ts
 * const pack = [
 * 	'https://cdn.ckeditor.com/ckeditor5/30.0.0/classic/ckeditor.js'
 * ];
 * ```
 */
type CKCdnResourcesBasicUrlsPack = Array<string>;
/**
 * A pack of resources to load (scripts and stylesheets) and the exported global variables.
 *
 * @example
 * ```ts
 * const pack = {
 * 	scripts: [
 * 		'https://cdn.ckeditor.com/ckeditor5/30.0.0/classic/ckeditor.js'
 * 	],
 * 	checkPluginLoaded: () => ( window as any ).ClassicEditor
 * };
 * ```
 */
export type CKCdnResourcesAdvancedPack<R> = {
    /**
     * List of resources to preload, it should improve the performance of loading the resources.
     */
    preload?: Array<string>;
    /**
     * List of scripts to load. Scripts are loaded in the order they are defined.
     */
    scripts?: Array<string | CKCdnScriptInjectorCallback>;
    /**
     * List of stylesheets to load. Stylesheets are loaded in the order they are defined.
     */
    stylesheets?: Array<string>;
    /**
     * Map of attributes to add to the script, stylesheet and link tags.
     * It can be used to specify `crossorigin` or `nonce` attributes on the injected HTML elements.
     */
    htmlAttributes?: Record<string, any>;
    /**
     * Callback that is executed before injecting the resources. It can be used to verify if the resources are already loaded.
     */
    beforeInject?: () => void;
    /**
     * Get JS object with global variables exported by scripts.
     */
    checkPluginLoaded?: () => Awaitable<R>;
};
/**
 * Callback that injects a script tag into the document.
 */
type CKCdnScriptInjectorCallback = (props: InjectScriptProps) => Awaitable<unknown>;
/**
 * Extracts the type of the exported global variables from a CKResourcesPack.
 */
export type InferCKCdnResourcesPackExportsType<P> = P extends CKCdnResourcesPack<infer R> ? R : never;
export {};
