'use client'

import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog'
import { DialogState } from '@/app/admin/categories/page'
import { z } from 'zod';
import { useEffect } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { Form, FormControl, FormField, FormItem, FormLabel } from '../ui/form';
import { Input } from '../ui/input';
import { Button } from '../ui/button';
import { createCategory, deleteCategoryById, updateCategoryById } from '@/app/api/categories';

type CategoryModelProps = {
    state: DialogState;
    onOpenChange: () => void;
    updateCategory: React.Dispatch<React.SetStateAction<any[]>>;
}

const formSchema = z.object({
    name: z.string().min(3, { message: "Category is Required" }),
    icon: z.string().min(3, { message: "Category is Required" })
})

const CategoryModel = ({ state, onOpenChange, updateCategory }: CategoryModelProps) => {
    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: { name: "", icon: "" }
    });

    // Reset form values when dialogState changes
    useEffect(() => {
        form.reset({ name: state.selectCategory?.name || "", icon: state.selectCategory?.icon || "" });
    }, [state.selectCategory, form]);

    const deleteCategory = async () => {
        if (!state.selectCategory?.id) return;
        try {
            const { success, error } = await deleteCategoryById(state.selectCategory.id);
            if (error) {
                console.error("Error deleting category:", error);
            } else {
                updateCategory(prev => prev.filter(cat => cat.id !== state.selectCategory?.id));
                onOpenChange(); // Close dialog after deletion
            }
        } catch (err) {
            console.error("Deletion error:", err);
        }
    };

    const onSubmit = async (values: z.infer<typeof formSchema>) => {
        if (state.action === "add") {
            try {
                const { category, error } = await createCategory(values);
                if (error) {
                    console.log(error);
                } else {
                    updateCategory(prev => [category, ...prev]);
                    onOpenChange(); // Close dialog on success
                }
            } catch (err) {
                console.error("Submission error:", err);
            }
        }
        if (state.action === "edit") {
            try {
                const { category, error } = await updateCategoryById(values, state.selectCategory?.id as string)
                if (error) {
                    console.log(error);
                } else {
                    updateCategory(prev => prev.map(cat =>
                        cat.id === category?.id ? category : cat // Replace updated category
                    ));
                    onOpenChange(); // Close dialog on success
                }
            } catch (err) {
                console.error("Submission error:", err);
            }
        }
    };


    return (
        <Dialog open={state.open} onOpenChange={onOpenChange}>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>
                        {state.action === "add" && "Add Category"}
                        {state.action === "edit" && "Edit Category"}
                        {state.action === "delete" && "Are you sure you want to delete this category?"}
                    </DialogTitle>
                    <DialogDescription>
                        {state.action === "delete" && "This action cannot be undone."}
                    </DialogDescription>
                </DialogHeader>

                {/* Show Form only when Adding or Editing */}
                {(state.action === "add" || state.action === "edit") && (
                    <Form {...form}>
                        <form onSubmit={form.handleSubmit(onSubmit)} className='space-y-6'>
                            <FormField
                                control={form.control}
                                name='name'
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Category Name</FormLabel>
                                        <FormControl>
                                            <Input type='text' placeholder='e.g. Web Development' {...field} />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name='icon'
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Category Icon</FormLabel>
                                        <FormControl>
                                            <Input type='text' placeholder='e.g. Globe' {...field} />
                                        </FormControl>
                                    </FormItem>
                                )}
                            />

                            <div className='flex items-center justify-end gap-2'>
                                <Button type='button' variant="outline" onClick={onOpenChange}>
                                    Cancel
                                </Button>
                                <Button type='submit'>
                                    {state.action === "edit" ? "Update Category" : "Add Category"}
                                </Button>
                            </div>
                        </form>
                    </Form>
                )}
                {
                    state.action === "delete" && (
                        <div className='flex items-center justify-end gap-2'>
                            <Button type='button' variant="outline" onClick={onOpenChange}>
                                Cancel
                            </Button>
                            <Button onClick={deleteCategory}>
                                Delete Category
                            </Button>
                        </div>
                    )
                }
            </DialogContent>
        </Dialog>
    )
}

export default CategoryModel;
