"use client"

import { createTestimonials } from '@/app/api/testimonials'
import { Button } from '@/components/ui/button'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { zodResolver } from '@hookform/resolvers/zod'
import React, { useEffect } from 'react'
import { useForm } from 'react-hook-form'
import { z } from "zod"
import { testimonials } from '@/components/admin/testimonials/columns'

const formSchema = z.object({
    name: z.string().min(2).max(50),
    designation: z.string().min(2, { message: "Designation is required" }),
    message: z.string().min(20, { message: "Message should be at least 20 characters" }),
    rating: z.string()
});

const TestimonialModel = ({
    open,
    onOpenChange,
    changeState,
    selectedTestimonial
}: {
    open: boolean,
    onOpenChange: (open: boolean) => void,
    changeState: React.Dispatch<React.SetStateAction<testimonials[]>>,
    selectedTestimonial: testimonials | null
}) => {



    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            name: "",
            designation: "",
            message: "",
            rating: ""
        }
    });

    // Pre-fill the form when editing
    useEffect(() => {
        if (isEditing) {
            form.reset(selectedTestimonial as any);
        }
    }, [selectedTestimonial, form]);

    // Handle Submit
    const onSubmit = async (values: z.infer<typeof formSchema>) => {
        if (isEditing) {
            // const { testimonials, error } = await updateTestimonials(selectedTestimonial!.id, values);
            // if (!error && testimonials) {
            //     changeState((prev) => prev.map(t => t.id === selectedTestimonial!.id ? testimonials : t));
            // }
        } else {
            const { testimonials, error } = await createTestimonials(values);
            if (!error && testimonials) {
                changeState((prev) => [...prev, testimonials]);
            }
        }
        onOpenChange(false);
    };

    // Handle Delete
    const handleDelete = async () => {
        // if (selectedTestimonial) {
        //     const { error } = await deleteTestimonial(selectedTestimonial.id);
        //     if (!error) {
        //         changeState((prev) => prev.filter(t => t.id !== selectedTestimonial.id));
        //         onOpenChange(false);
        //     }
        // }
    };

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>{isDeleting ? "Confirm Delete" : isEditing ? "Edit Testimonial" : "Add New Testimonial"}</DialogTitle>
                </DialogHeader>

                {isDeleting ? (
                    // Delete Confirmation
                    <div>
                        <p>Are you sure you want to delete this testimonial?</p>
                        <div className="flex justify-end space-x-4 mt-4">
                            <Button variant="outline" onClick={() => onOpenChange(false)}>Cancel</Button>
                            <Button variant="destructive" onClick={handleDelete}>Delete</Button>
                        </div>
                    </div>
                ) : (
                    // Add/Edit Form
                    <Form {...form}>
                        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                            <FormField
                                control={form.control}
                                name="name"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Name</FormLabel>
                                        <FormControl>
                                            <Input type="text" placeholder="e.g. Microsoft" {...field} />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name="designation"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Designation</FormLabel>
                                        <FormControl>
                                            <Input type="text" placeholder="e.g. Developer at BinaryQubit" {...field} />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name="message"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Message</FormLabel>
                                        <FormControl>
                                            <Textarea placeholder="e.g. Experience at BinaryQubit" {...field} />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <FormField
                                control={form.control}
                                name="rating"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Rating (0 - 5)</FormLabel>
                                        <FormControl>
                                            <Input type="text" placeholder="Rating should be 0 - 5" {...field} />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                            <div className="flex justify-end">
                                <Button type="submit">{isEditing ? "Update" : "Add"} Testimonial</Button>
                            </div>
                        </form>
                    </Form>
                )}
            </DialogContent>
        </Dialog>
    );
};

export default TestimonialModel;
