'use client'
import { DialogState } from '@/app/admin/courses/[id]/sections/page'
import { createsection, deletesectionById, updatesectionById } from '@/app/api/sections'
import { Button } from '@/components/ui/button'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form'
import { Input } from '@/components/ui/input'
import { zodResolver } from '@hookform/resolvers/zod'
import React, { useEffect } from 'react'
import { useForm } from 'react-hook-form'
import { z } from 'zod'

type SectionDialogProps = {
    state: DialogState,
    onOpenChange: () => void,
    updateSection: React.Dispatch<React.SetStateAction<any[]>>,
    courseId: string
}

const formSchema = z.object({
    title: z.string().min(3, { message: "Section title is required" })
})
const SectionDialog = ({ state, onOpenChange, updateSection, courseId }: SectionDialogProps) => {
    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            title: state.section?.title || ""
        }
    })

    // Reset form when section changes
    useEffect(() => {
        form.reset({
            title: state.section?.title || ""
        });
    }, [state.section, form]);
    const deletesection = async () => {
        if (!state.section?.id) return;
        try {
            const { success, error } = await deletesectionById(state.section.id);
            if (error) {
                console.log("Error deleting section:", error);
            } else {
                updateSection(prev => prev.filter(cat => cat.id !== state.section?.id));
                onOpenChange(); // Close dialog after deletion
            }
        } catch (err) {
            console.error("Deletion error:", err);
        }
    };

    const onSubmit = async (values: z.infer<typeof formSchema>) => {
        if (state.action === "add") {
            try {
                const { section, error } = await createsection(values, courseId);
                if (error) {
                    console.log(error);
                } else {
                    updateSection(prev => [...prev, section]);
                    onOpenChange(); // Close dialog on success
                }
            } catch (err) {
                console.error("Submission error:", err);
            }
        }
        if (state.action === "edit") {
            try {
                const { section, error } = await updatesectionById(values, state.section?.id as string, courseId)
                if (error) {
                    console.log(error);
                } else {
                    updateSection(prev => prev.map(cat =>
                        cat.id === section?.id ? section : cat // Replace updated section
                    ));
                    onOpenChange(); // Close dialog on success
                }
            } catch (err) {
                console.error("Submission error:", err);
            }
        }
    };
    return (
        <div>
            <Dialog open={state.open} onOpenChange={onOpenChange}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>
                            {state.action === 'add' && "Add Section"}
                            {state.action === 'edit' && "Edit Section"}
                            {state.action === 'delete' && "Delete Section"}
                        </DialogTitle>
                        <DialogDescription>
                            {state.action === 'delete' && " Are You Sure ?"}
                        </DialogDescription>
                    </DialogHeader>
                    {(state.action === 'add' || state.action === 'edit') &&
                        <Form {...form}>
                            <form onSubmit={form.handleSubmit(onSubmit)} className='space-y-4'>
                                <FormField
                                    control={form.control}
                                    name="title"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>Section Title</FormLabel>
                                            <FormControl>
                                                <Input type='text' placeholder='e.g. Introduction' {...field} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <div className='flex items-center justify-end gap-2'>
                                    <Button type='button' variant="outline" onClick={onOpenChange}>
                                        Cancel
                                    </Button>
                                    <Button type='submit'>
                                        {state.action === 'edit' ? "Update Course" : "Add Course"}
                                    </Button>
                                </div>

                            </form>

                        </Form>
                    }
                    {state.action === 'delete' &&
                        <div className='flex items-center justify-end gap-2'>
                            <Button type='button' variant="outline" onClick={onOpenChange}>
                                Cancel
                            </Button>
                            <Button onClick={deletesection}>
                                Delete
                            </Button>
                        </div>
                    }
                </DialogContent>
            </Dialog>
        </div>
    )
}

export default SectionDialog