'use client'

import { LessonDialogState } from '@/app/admin/courses/[id]/sections/[sectionId]/lessons/page'
import { createLesson, deleteLessonById, updateLessonById } from '@/app/api/lessons'

import { Button } from '@/components/ui/button'
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog'
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form'
import { Input } from '@/components/ui/input'
import { Textarea } from '@/components/ui/textarea'
import { zodResolver } from '@hookform/resolvers/zod'
import React, { useEffect } from 'react'
import { useForm } from 'react-hook-form'
import { z } from 'zod'

type LessonDialogProps = {
    state: LessonDialogState,
    onOpenChange: () => void,
    updateLesson: React.Dispatch<React.SetStateAction<any[]>>,
    sectionId: string
}

const formSchema = z.object({
    title: z.string().min(3, { message: "Section title is required" }),
    content: z.string().min(10, { message: "Content is required" }),
    videoUrl: z.string().optional()

})
const LessonDialog = ({ state, onOpenChange, updateLesson, sectionId }: LessonDialogProps) => {
    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            title: state.Lesson?.title || "",
            content: state.Lesson?.content || "",
            videoUrl: state.Lesson?.videoUrl || ""
        }
    })

    // Reset form when section changes
    useEffect(() => {
        form.reset({
            title: state.Lesson?.title || "",
            content: state.Lesson?.content || "",
            videoUrl: state.Lesson?.videoUrl || ""
        });
    }, [state.Lesson, form]);
    const deletesection = async () => {
        if (!state.Lesson?.id) return;
        try {
            const { success, error } = await deleteLessonById(state.Lesson.id);
            if (error) {
                console.log("Error deleting section:", error);
            } else {
                updateLesson(prev => prev.filter(cat => cat.id !== state.Lesson?.id));
                onOpenChange(); // Close dialog after deletion
            }
        } catch (err) {
            console.error("Deletion error:", err);
        }
    };

    const onSubmit = async (values: z.infer<typeof formSchema>) => {
        if (state.action === "add") {
            try {
                const { lesson, error } = await createLesson(values, sectionId);
                if (error) {
                    console.log(error);
                } else {
                    updateLesson(prev => [...prev, lesson]);
                    onOpenChange(); // Close dialog on success
                }
            } catch (err) {
                console.error("Submission error:", err);
            }
        }
        if (state.action === "edit") {
            try {
                const { lesson, error } = await updateLessonById(values, state.Lesson?.id as string)
                if (error) {
                    console.log(error);
                } else {
                    updateLesson(prev => prev.map(cat =>
                        cat.id === lesson?.id ? lesson : cat // Replace updated section
                    ));
                    onOpenChange(); // Close dialog on success
                }
            } catch (err) {
                console.error("Submission error:", err);
            }
        }
    };
    return (
        <div>
            <Dialog open={state.open} onOpenChange={onOpenChange}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>
                            {state.action === 'add' && "Add Lesson"}
                            {state.action === 'edit' && "Edit Lesson"}
                            {state.action === 'delete' && "Delete Lesson"}
                        </DialogTitle>
                        <DialogDescription>
                            {state.action === 'delete' && " Are You Sure ?"}
                        </DialogDescription>
                    </DialogHeader>
                    {(state.action === 'add' || state.action === 'edit') &&
                        <Form {...form}>
                            <form onSubmit={form.handleSubmit(onSubmit)} className='space-y-4'>
                                <FormField
                                    control={form.control}
                                    name="title"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>Lesson Title</FormLabel>
                                            <FormControl>
                                                <Input type='text' placeholder='e.g. Introduction' {...field} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <FormField
                                    control={form.control}
                                    name="content"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>Lesson Content</FormLabel>
                                            <FormControl>
                                                <Textarea placeholder='e.g.  This is Introduction' {...field} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />
                                <FormField
                                    control={form.control}
                                    name="videoUrl"
                                    render={({ field }) => (
                                        <FormItem>
                                            <FormLabel>Lesson Content</FormLabel>
                                            <FormControl>
                                                <Input type='text' placeholder='e.g.  https://youtube.com/' {...field} />
                                            </FormControl>
                                            <FormMessage />
                                        </FormItem>
                                    )}
                                />

                                <div className='flex items-center justify-end gap-2'>
                                    <Button type='button' variant="outline" onClick={onOpenChange}>
                                        Cancel
                                    </Button>
                                    <Button type='submit'>
                                        {state.action === 'edit' ? "Update Lesson" : "Add Lesson"}
                                    </Button>
                                </div>

                            </form>

                        </Form>
                    }
                    {state.action === 'delete' &&
                        <div className='flex items-center justify-end gap-2'>
                            <Button type='button' variant="outline" onClick={onOpenChange}>
                                Cancel
                            </Button>
                            <Button onClick={deletesection}>
                                Delete
                            </Button>
                        </div>
                    }
                </DialogContent>
            </Dialog>
        </div>
    )
}

export default LessonDialog