'use client'
import { DialogState } from '@/app/admin/courses/page';
import { createCourse, deleteCourseById, updateCourse } from '@/app/api/courses';
import { Button } from '@/components/ui/button';
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from '@/components/ui/command';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Textarea } from '@/components/ui/textarea';
import { cn } from '@/lib/utils';
import { zodResolver } from '@hookform/resolvers/zod';
import { Check, ChevronsUpDown } from 'lucide-react';
import React, { useEffect } from 'react'
import { useForm } from 'react-hook-form';
import { z } from 'zod';

type CourseModelProps = {
  state: DialogState;
  onOpenChange: () => void;
  updateState: React.Dispatch<React.SetStateAction<any[]>>;
  categories: { id: string, value: string }[];
}

const formSchema = z.object({
  name: z.string().min(3, { message: "Course Name is required" }),
  description: z.string().min(5, { message: "Description must be at least 5 characters" }),
  price: z.string().regex(/^\d+$/, { message: "Price must be a number" }),
  duration: z.string().regex(/^\d+$/, { message: "Duration must be a number" }),
  level: z.string(),
  categoryId: z.string().min(1, { message: "Category is required" }),
  banner: z.any()
})

const CourseModel = ({ state, onOpenChange, updateState, categories }: CourseModelProps) => {
  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      name: "",
      description: "",
      price: "",
      duration: "",
      level: "Beginner",
      categoryId: "",
      banner: ""
    }
  });

  // Update form values when `state.course` changes
  useEffect(() => {
    if (state.course) {
      form.reset({
        name: state.course.name,
        description: state.course.description,
        price: String(state.course.price),
        duration: String(state.course.duration),
        level: state.course.level,
        categoryId: state.course.category?.id || "",
      });
    }
  }, [state.course, form]);

  const deleteCourse = async () => {
    if (!state.course?.id) return;
    try {
      const { success, error } = await deleteCourseById(state.course.id);
      if (error) {
        console.error("Error deleting category:", error);
      } else {
        updateState(prev => prev.filter(cat => cat.id !== state.course?.id));
        onOpenChange(); // Close dialog after deletion
      }
    } catch (err) {
      console.error("Deletion error:", err);
    }
  };
  const onSubmit = async (values: z.infer<typeof formSchema>) => {
    console.log(values)
    if (state.action === 'edit' && state.course) {
      // Update Course
      const { course, error } = await updateCourse(state.course.id, values);
      if (!error) {
        updateState(prev => prev.map(c => (c.id === course?.id ? course : c)));
        onOpenChange(); // Close dialog
      }
    } else {
      // Create Course
      const { course, error } = await createCourse(values);
      if (!error) {
        updateState(prev => [...prev, course]);
        onOpenChange(); // Close dialog
      }
    }
  };

  // Handle Delete


  return (
    <Dialog open={state.open} onOpenChange={onOpenChange}>
      <ScrollArea className="max-h-[80vh] overflow-y-auto">
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {state.action === 'add' && "Add New Course"}
              {state.action === 'edit' && "Edit Course"}
              {state.action === 'delete' && "Are You Sure?"}
            </DialogTitle>

            <DialogDescription>{state.action === 'delete' && (<>This action cannot be undone.</>)}</DialogDescription>

          </DialogHeader>

          {state.action !== 'delete' ? (
            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className='space-y-6'>
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Course Name</FormLabel>
                      <FormControl>
                        <Input type='text' placeholder='e.g. Next.js' {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="description"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Course Description</FormLabel>
                      <FormControl>
                        <Textarea className="w-full h-32" placeholder='e.g. Learn Next.js from scratch' {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <div className='flex items-center justify-between gap-4'>
                  <FormField
                    control={form.control}
                    name="categoryId"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Course Category</FormLabel>
                        <Popover>
                          <PopoverTrigger asChild>
                            <FormControl>
                              <Button
                                variant="outline"
                                role="combobox"
                                className={cn(
                                  "w-[200px] justify-between",
                                  !field.value && "text-muted-foreground"
                                )}
                              >
                                {field.value
                                  ? categories.find((cat) => cat.id === field.value)?.value
                                  : "Select Category"}
                                <ChevronsUpDown className="opacity-50" />
                              </Button>
                            </FormControl>
                          </PopoverTrigger>
                          <PopoverContent className="w-[200px] p-0">
                            <Command>
                              <CommandInput placeholder="Search Category..." className="h-9" />
                              <CommandList>
                                <CommandEmpty>No Category found.</CommandEmpty>
                                <CommandGroup>
                                  {categories.map((cat) => (
                                    <CommandItem
                                      value={cat.id}
                                      key={cat.id}
                                      onSelect={() => form.setValue("categoryId", cat.id)}
                                    >
                                      {cat.value}
                                      <Check className={cn("ml-auto", cat.id === field.value ? "opacity-100" : "opacity-0")} />
                                    </CommandItem>
                                  ))}
                                </CommandGroup>
                              </CommandList>
                            </Command>
                          </PopoverContent>
                        </Popover>
                        <FormDescription>Select a category from the dropdown</FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="level"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Course Level</FormLabel>
                        <FormControl>
                          <select {...field}  className="border rounded-lg p-2 w-full">
                            <option value="Beginner">Beginner</option>
                            <option value="Professional">Professional</option>
                            <option value="Expert">Expert</option>
                          </select>
                        </FormControl>
                        <FormDescription>Select course difficulty level</FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <FormField
                  control={form.control}
                  name="price"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Course Price</FormLabel>
                      <FormControl>
                        <Input type="number" placeholder="e.g. 15000" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="duration"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Course Duration (Hours)</FormLabel>
                      <FormControl>
                        <Input type="number" placeholder="e.g. 180" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="banner"
                  render={({ field: { onChange, ...field } }) => (
                    <FormItem>
                      <FormLabel>Company logo</FormLabel>
                      <FormControl>
                        <Input type="file"
                          accept="image/*"
                          onChange={(e) => {
                            const file = e.target.files?.[0];
                            onChange(file);  // ✅ Store the File object
                          }} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <div className='flex items-center justify-end gap-2'>
                  <Button type='button' variant="outline" onClick={onOpenChange}>
                    Cancel
                  </Button>
                  <Button type='submit'>
                    {state.action === 'edit' ? "Update Course" : "Add Course"}
                  </Button>
                </div>
              </form>
            </Form>
          ) :
            <div className='flex items-center justify-end gap-2'>
              <Button type='button' variant="outline" onClick={onOpenChange}>
                Cancel
              </Button>
              <Button onClick={deleteCourse}>
                Delete Course
              </Button>
            </div>
          }
        </DialogContent>
      </ScrollArea>
    </Dialog>
  )
}

export default CourseModel;
